// to avoid redefinitions libubus was included here
#include <libubus.h>
#include <libtlt_uci.h>
#include <uci.h>

#include "ubus_server.h"
#include "utils_api.h"

int mrp_get(int *count, struct mrp_status *status);
int mrp_del(uint32_t br_ifindex, uint32_t ring_nr);
int mrp_add_v2(uint32_t br_ifindex, uint32_t ring_nr, uint32_t pport, uint32_t sport, uint32_t ring_role,
	       uint16_t prio, uint8_t ring_recv, uint8_t react_on_link_change, uint32_t in_role,
	       uint16_t in_id, uint32_t iport, uint32_t in_mode, uint8_t in_recv, uint32_t cfm_instance,
	       uint32_t cfm_level, uint32_t cfm_mepid, uint32_t cfm_peer_mepid, char *cfm_maid,
	       char *cfm_dmac, uint8_t *domain, char *domain_name, struct mrp_params *prm);
int mrp_add_tmp(char *br_ifname, uint32_t ring_nr, char *pport_ifname, char *sport_ifname, uint32_t ring_role,
		uint16_t prio, uint8_t ring_recv, uint8_t react_on_link_change, uint32_t in_role,
		uint16_t in_id, uint32_t iport, uint32_t in_mode, uint8_t in_recv, uint32_t cfm_instance,
		uint32_t cfm_level, uint32_t cfm_mepid, uint32_t cfm_peer_mepid, char *cfm_maid,
		char *cfm_dmac, uint8_t *domain, char *domain_name, struct mrp_params *prm);
void mrp_set_domain(uint32_t ifindex, uint32_t ring_nr, uint8_t *domain);
void mrp_set_domain_name(uint32_t ifindex, uint32_t ring_nr, char *domain_name);
int mrp_flush_fdb(uint32_t br_ifindex, uint32_t ring_nr);

enum {
	UBUS_RING_ID,
	UBUS_RING_BRIDGE,
	UBUS_RING_PPORT,
	UBUS_RING_SPORT,
	UBUS_RING_ROLE,
	UBUS_RING_PRIORITY,
	UBUS_RING_RECOVERY_TIME,
	UBUS_DOMAIN_UUID,
	UBUS_DOMAIN_NAME,
	UBUS_PRM_TOP_CHG_T,
	UBUS_PRM_TOP_NR_MAX,
	UBUS_PRM_TST_SHORT_T,
	UBUS_PRM_TST_DEFAULT_T,
	UBUS_PRM_TST_NR_MAX,
	UBUS_PRM_TST_EXT_NRMA,
	UBUS_PRM_LNK_DOWN_T,
	UBUS_PRM_LNK_UP_T,
	UBUS_PRM_LNK_NR_MAX,
	UBUS_AUTO_ENABLE,
	UBUS_UPDATE_CONF,
	UBUS_RING_MAX
};

static struct ubus_context ubus_ctx = { 0 };
static ev_io ubus_watcher;
static struct uci_context *uci_ctx = NULL;

static int ubus_getmrp_cb(struct ubus_context *ctx, struct ubus_object *obj, struct ubus_request_data *req,
			  const char *method, struct blob_attr *msg);

static int ubus_delmrp_cb(struct ubus_context *ctx, struct ubus_object *obj, struct ubus_request_data *req,
			  const char *method, struct blob_attr *msg);

static int ubus_addmrp_cb(struct ubus_context *ctx, struct ubus_object *obj, struct ubus_request_data *req,
			  const char *method, struct blob_attr *msg);

static int ubus_getprm_cb(struct ubus_context *ctx, struct ubus_object *obj, struct ubus_request_data *req,
			  const char *method, struct blob_attr *msg);

static int ubus_flush_mrp_fdb_cb(struct ubus_context *ctx, struct ubus_object *obj,
				 struct ubus_request_data *req, const char *method, struct blob_attr *msg);

static int valid_ring_recv(uint32_t recovery_time);
static void getprm_fill_obj(struct blob_buf *b, struct blob_attr **tb, struct mrp_status *mrp, int count,
			    int br, int ring_nr);

static const struct blobmsg_policy ubus_getmrp_policy[] = {
	[UBUS_RING_ID]	   = { .name = "ring_nr", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_RING_BRIDGE] = { .name = "bridge", .type = BLOBMSG_TYPE_STRING },
	[UBUS_UPDATE_CONF] = { .name = "update_config", .type = BLOBMSG_TYPE_BOOL },
};

static const struct blobmsg_policy ubus_addmrp_policy[] = {
	[UBUS_RING_ID]		  = { .name = "ring_nr", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_RING_BRIDGE]	  = { .name = "bridge", .type = BLOBMSG_TYPE_STRING },
	[UBUS_RING_ROLE]	  = { .name = "ring_role", .type = BLOBMSG_TYPE_STRING },
	[UBUS_RING_PRIORITY]	  = { .name = "prio", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_RING_PPORT]	  = { .name = "pport", .type = BLOBMSG_TYPE_STRING },
	[UBUS_RING_SPORT]	  = { .name = "sport", .type = BLOBMSG_TYPE_STRING },
	[UBUS_RING_RECOVERY_TIME] = { .name = "ring_recv", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_DOMAIN_UUID]	  = { .name = "domain", .type = BLOBMSG_TYPE_ARRAY },
	[UBUS_DOMAIN_NAME]	  = { .name = "domain_name", .type = BLOBMSG_TYPE_STRING },
	[UBUS_PRM_TOP_CHG_T]	  = { .name = "TOPchgT", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_PRM_TOP_NR_MAX]	  = { .name = "TOPNRmax", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_PRM_TST_SHORT_T]	  = { .name = "TSTshortT", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_PRM_TST_DEFAULT_T]  = { .name = "TSTdefaultT", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_PRM_TST_NR_MAX]	  = { .name = "TSTNRmax", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_PRM_TST_EXT_NRMA]	  = { .name = "TSTExtNRma", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_PRM_LNK_DOWN_T]	  = { .name = "LNKdownT", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_PRM_LNK_UP_T]	  = { .name = "LNKupT", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_PRM_LNK_NR_MAX]	  = { .name = "LNKNRmax", .type = BLOBMSG_TYPE_INT32 },
	[UBUS_AUTO_ENABLE]	  = { .name = "auto_enable", .type = BLOBMSG_TYPE_BOOL },
	[UBUS_UPDATE_CONF]	  = { .name = "update_config", .type = BLOBMSG_TYPE_BOOL },
};

static const struct ubus_method mrp_methods[] = {
	UBUS_METHOD("getmrp", ubus_getmrp_cb, ubus_getmrp_policy),
	UBUS_METHOD("delmrp", ubus_delmrp_cb, ubus_getmrp_policy),
	UBUS_METHOD("addmrp", ubus_addmrp_cb, ubus_addmrp_policy),
	UBUS_METHOD("getprm", ubus_getprm_cb, ubus_getmrp_policy),
	UBUS_METHOD("fdbmrp", ubus_flush_mrp_fdb_cb, ubus_getmrp_policy),
};

static struct ubus_object_type mrp_object_type = UBUS_OBJECT_TYPE("mrp", mrp_methods);

static struct ubus_object ubus_mrp_object = {
	.name	   = "mrp",
	.type	   = &mrp_object_type,
	.methods   = mrp_methods,
	.n_methods = ARRAY_SIZE(mrp_methods),
};

static void getmrp_fill_array(struct blob_buf *b, struct blob_attr **tb, struct mrp_status *status, int count,
			      uint16_t ring_id, int br);

static void ubus_watcher_cb(EV_P_ ev_io *w, int revents)
{
	if (!ubus_ctx.sock.cb) {
		return;
	}

	ubus_ctx.sock.cb(&ubus_ctx.sock, revents);
}

int ubus_init()
{
	int rc = ubus_connect_ctx(&ubus_ctx, NULL);
	if (rc) {
		fprintf(stderr, "Failed to connect ubus context, %s\n", ubus_strerror(errno));
		return -1;
	}

	rc = ubus_add_object(&ubus_ctx, &ubus_mrp_object);
	if (rc) {
		fprintf(stderr, "Failed to add ubus object: %s\n", ubus_strerror(rc));
		goto err;
	}

	uci_ctx = uci_init();
	if (!uci_ctx) {
		fprintf(stderr, "Failed to allocate uci context\n");
		rc = -1;
		goto err;
	}

	ev_io_init(&ubus_watcher, &ubus_watcher_cb, ubus_ctx.sock.fd, EV_READ);
	ev_io_start(EV_DEFAULT, &ubus_watcher);

	return 0;
err:
	ubus_shutdown(&ubus_ctx);
	return rc;
}

void ubus_cleanup()
{
	ubus_shutdown(&ubus_ctx);
	uci_free_context(uci_ctx);
}

static void getmrp_fill_array(struct blob_buf *b, struct blob_attr **tb, struct mrp_status *status, int count,
			     uint16_t ring_id, int br)
{
	for (int i = 0; i < count; i++) {
		void *table		 = NULL;
		void *array = NULL;
		char ifname[IF_NAMESIZE] = { 0 };

		if (tb[UBUS_RING_ID] && ring_id != status[i].ring_nr) {
			continue;
		}

		if (tb[UBUS_RING_BRIDGE] && br != status[i].br) {
			continue;
		}

		table = blobmsg_open_table(b, NULL);
		blobmsg_add_u32(b, "ring_nr", status[i].ring_nr);
		blobmsg_add_string(b, "bridge", if_indextoname(status[i].br, ifname));
		blobmsg_add_string(b, "pport", if_indextoname(status[i].pport, ifname));
		blobmsg_add_string(b, "sport", if_indextoname(status[i].sport, ifname));
		blobmsg_add_u32(b, "mra_support", status[i].mra_support);
		blobmsg_add_string(b, "ring_role", ring_role_str(status[i].ring_role));
		blobmsg_add_u32(b, "prio", status[i].prio);
		blobmsg_add_u32(b, "ring_recv", ring_recv_int(status[i].ring_recv));
		blobmsg_add_string(b, "domain_name", status[i].domain_name);

		array = blobmsg_open_array(b, "domain");
		for (int j = 0; j < MRP_DOMAIN_UUID_LENGTH; j++) {
			blobmsg_add_u32(b, NULL, status[i].domain[j]);
		}
		blobmsg_close_array(b, array);

		if (status[i].ring_role == BR_MRP_RING_ROLE_MRM) {
			blobmsg_add_string(b, "ring_state", mrm_state_str(status[i].ring_state));
		}
		if (status[i].ring_role == BR_MRP_RING_ROLE_MRC) {
			blobmsg_add_string(b, "ring_state", mrc_state_str(status[i].ring_state));
		}

		blobmsg_add_string(b, "pport_state", port_state_to_str(status[i].pport_state));
		blobmsg_add_string(b, "sport_state", port_state_to_str(status[i].sport_state));

		if (status[i].in_role == BR_MRP_IN_ROLE_DISABLED) {
			blobmsg_close_table(b, table);
			continue;
		}

		blobmsg_add_string(b, "iport", if_indextoname(status[i].iport, ifname));
		blobmsg_add_u32(b, "in_id", status[i].in_id);
		blobmsg_add_string(b, "in_role", in_role_str(status[i].in_role));
		blobmsg_add_string(b, "in_recv", in_recv_str(status[i].in_recv));
		blobmsg_add_string(b, "in_mode", in_mode_str(status[i].in_mode));
		if (status[i].in_role == BR_MRP_IN_ROLE_MIM)
			blobmsg_add_string(b, "in_state", mim_state_str(status[i].in_state));
		if (status[i].in_role == BR_MRP_IN_ROLE_MIC)
			blobmsg_add_string(b, "in_state", mic_state_str(status[i].in_state));

		blobmsg_close_table(b, table);
	}
}

static int ubus_flush_mrp_fdb_cb(struct ubus_context *ctx, struct ubus_object *obj,
				 struct ubus_request_data *req, const char *method, struct blob_attr *msg)
{
	int rc = UBUS_STATUS_OK, br = 0;
	uint16_t ring_id			    = 0;
	struct blob_attr *tb[UBUS_RING_MAX]	    = { 0 };
	struct blob_buf b			    = { 0 };
	char *ubus_error_msg			    = NULL;
	errno = 0;

	if (blob_buf_init(&b, 0)) {
		fprintf(stderr, "blob_bug_init failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

	if (blobmsg_parse(ubus_getmrp_policy, ARRAY_SIZE(ubus_getmrp_policy), tb, blob_data(msg),
			  blob_len(msg))) {
		ubus_error_msg = "Failed to parse ubus message\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto clean;
	}

	if (tb[UBUS_RING_ID]) {
		ring_id = (uint16_t)blobmsg_get_u32(tb[UBUS_RING_ID]);
	} else {
		ubus_error_msg = "Missing ring number argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (tb[UBUS_RING_BRIDGE]) {
		br = if_nametoindex(blobmsg_data(tb[UBUS_RING_BRIDGE]));
		if (errno) {
			ubus_error_msg = strerror(errno);
			fprintf(stderr, "%s\n", ubus_error_msg);
			goto end;
		}
	} else {
		ubus_error_msg = "Missing ring bridge argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (mrp_flush_fdb(br, ring_id)) {
		ubus_error_msg = "Failed to clear fdb from mrp instances\n";
		fprintf(stderr, "%s\n", ubus_error_msg);
		goto end;
	}

	blobmsg_add_u8(&b, "success", true);

end:
	if (ubus_error_msg && blobmsg_add_string(&b, "error", ubus_error_msg)) {
		fprintf(stderr, "blobmsg_add_string failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

	if (ubus_send_reply(&ubus_ctx, req, b.head)) {
		fprintf(stderr, "ubus_send_reply failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

clean:
	blob_buf_free(&b);
	return rc;
}

static int ubus_getmrp_cb(struct ubus_context *ctx, struct ubus_object *obj, struct ubus_request_data *req,
			  const char *method, struct blob_attr *msg)
{
	int count = 0, rc = UBUS_STATUS_OK, br = 0;
	uint16_t ring_id			    = 0;
	struct mrp_status status[MAX_MRP_INSTANCES] = { 0 };
	struct blob_attr *tb[UBUS_RING_MAX]	    = { 0 };
	struct blob_buf b			    = { 0 };
	char *ubus_error_msg			    = NULL;
	void *blob_array;
	errno = 0;

	if (blob_buf_init(&b, 0)) {
		fprintf(stderr, "blob_bug_init failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

	if (blob_len(msg)) {
		if (blobmsg_parse(ubus_getmrp_policy, ARRAY_SIZE(ubus_getmrp_policy), tb, blob_data(msg),
				  blob_len(msg))) {
			ubus_error_msg = "Failed to parse ubus message\n";
			fprintf(stderr, "%s", ubus_error_msg);
			rc = UBUS_STATUS_UNKNOWN_ERROR;
			goto clean;
		}
	}

	if (tb[UBUS_RING_ID]) {
		ring_id = (uint16_t)blobmsg_get_u32(tb[UBUS_RING_ID]);
	}

	if (tb[UBUS_RING_BRIDGE]) {
		br = if_nametoindex(blobmsg_data(tb[UBUS_RING_BRIDGE]));
		if (errno) {
			ubus_error_msg = strerror(errno);
			fprintf(stderr, "%s\n", ubus_error_msg);
			goto end;
		}
	}

	if (mrp_get(&count, status)) {
		ubus_error_msg = "Failed to get mrp_instances\n";
		fprintf(stderr, "%s\n", ubus_error_msg);
		goto end;
	}

	blob_array = blobmsg_open_array(&b, "mrp_instances");

	getmrp_fill_array(&b, tb, status, count, ring_id, br);

	blobmsg_close_array(&b, blob_array);
end:
	if (ubus_error_msg && blobmsg_add_string(&b, "error", ubus_error_msg)) {
		fprintf(stderr, "blobmsg_add_string failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

	if (ubus_send_reply(&ubus_ctx, req, b.head)) {
		fprintf(stderr, "ubus_send_reply failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

clean:
	blob_buf_free(&b);
	return rc;
}

static int uci_delete_cfg(struct uci_context *ctx, struct uci_package *pkg, struct mrp_status *mrp,
			  int mrp_count)
{
	int rc				= 0;
	int section_ring_nr		= 0;
	struct uci_element *e		= NULL;
	struct uci_section *uci_section = NULL;

	uci_foreach_element (&pkg->sections, e) {
		uci_section = uci_to_section(e);
		if (!strcmp(uci_section->type, "mrp")) {
			section_ring_nr = ucix_get_option_int(ctx, "mrp", uci_section->e.name, "ring_nr", -1);
			bool found	= false;
			for (size_t i = 0; i < mrp_count; i++) {
				if (mrp[i].ring_nr == section_ring_nr) {
					found = true;
					break;
				}
			}
			if (!found && section_ring_nr != 1) {
				rc |= ucix_delete(ctx, "mrp", uci_section->e.name, NULL);
			}
		}
	}

	return rc;
}

static int uci_add_cfg(struct uci_context *ctx, struct mrp_status *mrp, int mrp_count)
{
	int rc			 = 0;
	char section_name[8]	 = { 0 };
	char ring_nr_str[8]	 = { 0 };
	char ifname[IF_NAMESIZE] = { 0 };
	bool ring_nr_1_exists	 = false;

	for (size_t i = 0; i < mrp_count; i++) {
		if (mrp[i].ring_nr == 1) {
			ring_nr_1_exists = true;
			break;
		}
	}

	for (int i = 0; i < mrp_count; i++) {
		snprintf(ring_nr_str, sizeof(ring_nr_str), "%d", mrp[i].ring_nr);

		// if ring_nr is 1 and ring_nr 1 does not exist in the list, then use ring_nr 1
		snprintf(section_name, sizeof(section_name), "%d",
			 (i == 0 && !ring_nr_1_exists) ? 1 : mrp[i].ring_nr);

		// if we move instance to the first section, then delete the old one if it exists
		if (i == 0 && !ring_nr_1_exists && ucix_get_option(ctx, "mrp", ring_nr_str, NULL)) {
			rc |= ucix_delete(ctx, "mrp", ring_nr_str, NULL);
		}

		// create section if missing
		if (!ucix_get_option(ctx, "mrp", section_name, NULL)) {
			rc |= ucix_add_option(ctx, "mrp", section_name, NULL, "mrp");
		}

		rc |= ucix_add_option_int(ctx, "mrp", section_name, "enabled", 1);
		rc |= ucix_add_option_int(ctx, "mrp", section_name, "ring_nr", mrp[i].ring_nr);
		rc |= ucix_add_option(ctx, "mrp", section_name, "bridge", if_indextoname(mrp[i].br, ifname));
		rc |= ucix_add_option(ctx, "mrp", section_name, "pport",
				      if_indextoname(mrp[i].pport, ifname));
		rc |= ucix_add_option(ctx, "mrp", section_name, "sport",
				      if_indextoname(mrp[i].sport, ifname));
		rc |= ucix_add_option(ctx, "mrp", section_name, "role", ring_role_str(mrp[i].ring_role));
		rc |= ucix_add_option_int(ctx, "mrp", section_name, "ring_recv",
					  ring_recv_int(mrp[i].ring_recv));

		memset(section_name, 0, sizeof(section_name));
		memset(ring_nr_str, 0, sizeof(ring_nr_str));
	}

	return rc;
}

static int uci_update_cfg(struct uci_context *ctx)
{
	int rc					 = -1;
	int mrp_count				 = 0;
	struct mrp_status mrp[MAX_MRP_INSTANCES] = { 0 };
	struct uci_ptr ptr;

	if (mrp_get(&mrp_count, mrp)) {
		fprintf(stderr, "Failed to get mrp instances\n");
		goto end;
	}

	if (uci_lookup_ptr(ctx, &ptr, "mrp", false)) {
		fprintf(stderr, "Failed to load mrp configuration file.\n");
		goto end;
	}

	if (uci_delete_cfg(ctx, ptr.p, mrp, mrp_count)) {
		fprintf(stderr, "Failed to delete mrp instances\n");
		goto end;
	}

	if (uci_add_cfg(ctx, mrp, mrp_count)) {
		fprintf(stderr, "Failed to add mrp instances\n");
		goto end;
	}

	rc = ucix_commit(ctx, "mrp");
end:
	return rc;
}

static int ubus_delmrp_cb(struct ubus_context *ctx, struct ubus_object *obj, struct ubus_request_data *req,
			  const char *method, struct blob_attr *msg)
{
	int rc = UBUS_STATUS_OK, br = 0;
	uint16_t ring_id		    = 0;
	struct blob_attr *tb[UBUS_RING_MAX] = { 0 };
	struct blob_buf b		    = { 0 };
	char *ubus_error_msg		    = NULL;
	errno				    = 0;
	bool update_config		    = false;

	if (blob_buf_init(&b, 0)) {
		fprintf(stderr, "blob_bug_init failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

	if (blobmsg_parse(ubus_getmrp_policy, ARRAY_SIZE(ubus_getmrp_policy), tb, blob_data(msg),
			  blob_len(msg))) {
		ubus_error_msg = "Failed to parse ubus message\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto clean;
	}

	if (tb[UBUS_RING_ID]) {
		ring_id = (uint16_t)blobmsg_get_u32(tb[UBUS_RING_ID]);
	} else {
		ubus_error_msg = "Missing ring number argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (tb[UBUS_RING_BRIDGE]) {
		br = if_nametoindex(blobmsg_data(tb[UBUS_RING_BRIDGE]));
		if (errno) {
			ubus_error_msg = strerror(errno);
			fprintf(stderr, "%s\n", ubus_error_msg);
			goto end;
		}
	} else {
		ubus_error_msg = "Missing ring bridge argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (tb[UBUS_UPDATE_CONF]) {
		update_config = blobmsg_get_u8(tb[UBUS_UPDATE_CONF]);
	} // optional

	if (mrp_del(br, ring_id)) {
		ubus_error_msg = "Failed to delete mrp instances\n";
		fprintf(stderr, "%s\n", ubus_error_msg);
		goto end;
	}

	blobmsg_add_u8(&b, "success", true);

	if (update_config && uci_update_cfg(uci_ctx)) {
		ubus_error_msg = "Failed to update uci config";
		fprintf(stderr, "%s\n", ubus_error_msg);
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto end;
	}

end:
	if (ubus_error_msg && blobmsg_add_string(&b, "error", ubus_error_msg)) {
		fprintf(stderr, "blobmsg_add_string failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

	if (ubus_send_reply(&ubus_ctx, req, b.head)) {
		fprintf(stderr, "ubus_send_reply failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

clean:
	blob_buf_free(&b);
	return rc;
}

static int parse_domain(uint8_t *domain, struct blob_attr **tb)
{
	int rc		      = 1;
	struct blob_attr *cur = NULL;
	int rem		      = 0;

	if (tb[UBUS_DOMAIN_UUID] && blobmsg_data_len(tb[UBUS_DOMAIN_UUID])) {
		int i = 0;
		blobmsg_for_each_attr (cur, tb[UBUS_DOMAIN_UUID], rem) {
			domain[i] = (uint8_t)blobmsg_get_u32(cur);
			i++;
		}

		pr_info("domain: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X",
			domain[0], domain[1], domain[2], domain[3], domain[4], domain[5], domain[6],
			domain[7], domain[8], domain[9], domain[10], domain[11], domain[12], domain[13],
			domain[14], domain[15]);
		rc = 0;
	}

	return rc;
}

static int ubus_addmrp_cb(struct ubus_context *ctx, struct ubus_object *obj, struct ubus_request_data *req,
			  const char *method, struct blob_attr *msg)
{
	char *ubus_error_msg		    = NULL;
	errno				    = 0;
	int rc				    = UBUS_STATUS_OK;
	int is_domain = 0, auto_enable = 0;
	struct blob_attr *tb[UBUS_RING_MAX] = { 0 };
	struct blob_buf b		    = { 0 };
	uint16_t ring_id		    = 0;
	uint32_t pport = 0, sport = 0, recovery_time = 0, br = 0, role = 0, prio = 0;
	uint8_t domain[MRP_DOMAIN_UUID_LENGTH] = { 0 };
	char *domain_name		       = NULL;
	struct mrp_params prm		       = { 0 };
	char *br_ifname = NULL, *pport_ifname = NULL, *sport_ifname = NULL;
	bool update_config = false;

	if (blob_buf_init(&b, 0)) {
		fprintf(stderr, "blob_bug_init failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

	if (blobmsg_parse(ubus_addmrp_policy, ARRAY_SIZE(ubus_addmrp_policy), tb, blob_data(msg),
			  blob_len(msg))) {
		ubus_error_msg = "Failed to parse ubus message\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto clean;
	}

	if (tb[UBUS_AUTO_ENABLE]) {
		auto_enable = blobmsg_get_u8(tb[UBUS_AUTO_ENABLE]);
	}

	if (tb[UBUS_RING_ID]) {
		ring_id = (uint16_t)blobmsg_get_u32(tb[UBUS_RING_ID]);
	} else {
		ubus_error_msg = "Missing ring number argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (tb[UBUS_RING_BRIDGE]) {
		br_ifname = blobmsg_data(tb[UBUS_RING_BRIDGE]);
		br	  = auto_enable ? 0 : if_nametoindex(br_ifname);
		if (errno) {
			ubus_error_msg = strerror(errno);
			fprintf(stderr, "%s\n", ubus_error_msg);
			rc = UBUS_STATUS_UNKNOWN_ERROR;
			goto end;
		}
	} else {
		ubus_error_msg = "Missing ring bridge argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (tb[UBUS_RING_PPORT]) {
		pport_ifname = blobmsg_data(tb[UBUS_RING_PPORT]);
		pport	     = auto_enable ? 0 : if_nametoindex(pport_ifname);
		if (errno) {
			ubus_error_msg = strerror(errno);
			fprintf(stderr, "%s\n", ubus_error_msg);
			rc = UBUS_STATUS_UNKNOWN_ERROR;
			goto end;
		}
	} else {
		ubus_error_msg = "Missing ring primery port argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (tb[UBUS_RING_SPORT]) {
		sport_ifname = blobmsg_data(tb[UBUS_RING_SPORT]);
		sport	     = auto_enable ? 0 : if_nametoindex(sport_ifname);
		if (errno) {
			ubus_error_msg = strerror(errno);
			fprintf(stderr, "%s\n", ubus_error_msg);
			rc = UBUS_STATUS_UNKNOWN_ERROR;
			goto end;
		}
	} else {
		ubus_error_msg = "Missing ring secondary port argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (tb[UBUS_RING_ROLE]) {
		role = ring_role_int((char *)blobmsg_data(tb[UBUS_RING_ROLE]));
	} else {
		ubus_error_msg = "Missing ring role argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (tb[UBUS_RING_RECOVERY_TIME]) {
		recovery_time = blobmsg_get_u32(tb[UBUS_RING_RECOVERY_TIME]);

		if (valid_ring_recv(recovery_time)) {
			ubus_error_msg = "Invalid ring recovery time argument\n";
			fprintf(stderr, "%s", ubus_error_msg);
			rc = UBUS_STATUS_INVALID_ARGUMENT;
			goto end;
		}
	} // optional

	if (tb[UBUS_RING_PRIORITY]) {
		prio = blobmsg_get_u32(tb[UBUS_RING_PRIORITY]);
	} // optional

	int owr_prm = 0;
	if (tb[UBUS_PRM_TOP_CHG_T] && tb[UBUS_PRM_TOP_NR_MAX] && tb[UBUS_PRM_TST_SHORT_T] &&
	    tb[UBUS_PRM_TST_DEFAULT_T] && tb[UBUS_PRM_TST_NR_MAX] && tb[UBUS_PRM_TST_EXT_NRMA] &&
	    tb[UBUS_PRM_LNK_UP_T] && tb[UBUS_PRM_LNK_DOWN_T] && tb[UBUS_PRM_LNK_NR_MAX]) {
		prm.ring_topo_conf_interval	 = blobmsg_get_u32(tb[UBUS_PRM_TOP_CHG_T]);
		prm.ring_topo_conf_max		 = blobmsg_get_u32(tb[UBUS_PRM_TOP_NR_MAX]);
		prm.ring_test_conf_short	 = blobmsg_get_u32(tb[UBUS_PRM_TST_SHORT_T]);
		prm.ring_test_conf_interval	 = blobmsg_get_u32(tb[UBUS_PRM_TST_DEFAULT_T]);
		prm.ring_test_conf_max		 = blobmsg_get_u32(tb[UBUS_PRM_TST_NR_MAX]);
		prm.ring_test_conf_ext_max	 = blobmsg_get_u32(tb[UBUS_PRM_TST_EXT_NRMA]);
		prm.ring_link_conf_interval_up	 = blobmsg_get_u32(tb[UBUS_PRM_LNK_UP_T]);
		prm.ring_link_conf_interval_down = blobmsg_get_u32(tb[UBUS_PRM_LNK_DOWN_T]);
		prm.ring_link_conf_max		 = blobmsg_get_u32(tb[UBUS_PRM_LNK_NR_MAX]);
		owr_prm				 = 1;
	}

	if (!parse_domain(domain, tb)) {
		is_domain = 1;
	} // optional

	if (tb[UBUS_UPDATE_CONF]) {
		update_config = blobmsg_get_u8(tb[UBUS_UPDATE_CONF]);
	} // optional

	if (tb[UBUS_DOMAIN_NAME]) {
		domain_name = blobmsg_get_string(tb[UBUS_DOMAIN_NAME]);
	} // optional

	if (!auto_enable &&
	    mrp_add_v2(br, ring_id, pport, sport, role, prio, int_ring_recv(recovery_time), 0, 0, 0, 0, 0, 0, 0,
		       0, 0, 0, NULL, NULL, is_domain ? domain : NULL, domain_name, owr_prm ? &prm : NULL)) {
		ubus_error_msg = "Failed to add mrp instances\n";
		fprintf(stderr, "%s\n", ubus_error_msg);
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto end;
	}

	if (auto_enable && mrp_add_tmp(br_ifname, ring_id, pport_ifname, sport_ifname, role, prio,
				       int_ring_recv(recovery_time), 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL,
				       is_domain ? domain : NULL, domain_name, owr_prm ? &prm : NULL)) {
		ubus_error_msg = "Failed to add mrp instances\n";
		fprintf(stderr, "%s\n", ubus_error_msg);
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto end;
	}

	blobmsg_add_u8(&b, "success", true);

	if (update_config && uci_update_cfg(uci_ctx)) {
		ubus_error_msg = "Failed to update uci config";
		fprintf(stderr, "%s\n", ubus_error_msg);
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto end;
	}

end:
	if (ubus_error_msg && blobmsg_add_string(&b, "error", ubus_error_msg)) {
		fprintf(stderr, "blobmsg_add_string failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

	if (ubus_send_reply(&ubus_ctx, req, b.head)) {
		fprintf(stderr, "ubus_send_reply failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

clean:
	blob_buf_free(&b);
	return rc;
}

static void getprm_fill_obj(struct blob_buf *b, struct blob_attr **tb, struct mrp_status *mrp, int count,
			    int br, int ring_nr)
{
	for (int i = 0; i < count; i++) {
		if (mrp[i].br != br || mrp[i].ring_nr != ring_nr) {
			continue;
		}

		blobmsg_add_u32(b, "TOPchgT", mrp[i].ring_topo_conf_interval);
		blobmsg_add_u32(b, "TOPNRmax", mrp[i].ring_topo_conf_max);
		blobmsg_add_u32(b, "TSTshortT", mrp[i].ring_test_conf_short);
		blobmsg_add_u32(b, "TSTdefaultT", mrp[i].ring_test_conf_interval);
		blobmsg_add_u32(b, "TSTNRmax", mrp[i].ring_test_conf_max);
		blobmsg_add_u32(b, "TSTExtNRma", mrp[i].ring_test_conf_ext_max);
		blobmsg_add_u32(b, "LNKdownT", mrp[i].ring_link_conf_interval_down);
		blobmsg_add_u32(b, "LNKupT", mrp[i].ring_link_conf_interval_up);
		blobmsg_add_u32(b, "LNKNRmax", mrp[i].ring_link_conf_max);
	}
}

static int ubus_getprm_cb(struct ubus_context *ctx, struct ubus_object *obj, struct ubus_request_data *req,
			  const char *method, struct blob_attr *msg)
{
	int count = 0, rc = UBUS_STATUS_OK, br = 0;
	uint16_t ring_id			 = 0;
	struct blob_attr *tb[UBUS_RING_MAX]	 = { 0 };
	struct blob_buf b			 = { 0 };
	char *ubus_error_msg			 = NULL;
	struct mrp_status mrp[MAX_MRP_INSTANCES] = { 0 };
	errno					 = 0;

	if (blob_buf_init(&b, 0)) {
		fprintf(stderr, "blob_bug_init failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

	if (blob_len(msg)) {
		if (blobmsg_parse(ubus_getmrp_policy, ARRAY_SIZE(ubus_getmrp_policy), tb, blob_data(msg),
				  blob_len(msg))) {
			ubus_error_msg = "Failed to parse ubus message\n";
			fprintf(stderr, "%s", ubus_error_msg);
			rc = UBUS_STATUS_UNKNOWN_ERROR;
			goto clean;
		}
	}

	if (tb[UBUS_RING_ID]) {
		ring_id = (uint16_t)blobmsg_get_u32(tb[UBUS_RING_ID]);
	} else {
		ubus_error_msg = "Missing ring number argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (tb[UBUS_RING_BRIDGE]) {
		br = if_nametoindex(blobmsg_data(tb[UBUS_RING_BRIDGE]));
		if (errno) {
			ubus_error_msg = strerror(errno);
			fprintf(stderr, "%s\n", ubus_error_msg);
			goto end;
		}
	} else {
		ubus_error_msg = "Missing ring bridge argument\n";
		fprintf(stderr, "%s", ubus_error_msg);
		rc = UBUS_STATUS_INVALID_ARGUMENT;
		goto end;
	}

	if (mrp_get(&count, mrp)) {
		ubus_error_msg = "Failed to get mrp_instances\n";
		fprintf(stderr, "%s\n", ubus_error_msg);
		goto end;
	}

	getprm_fill_obj(&b, tb, mrp, count, br, ring_id);

end:
	if (ubus_error_msg && blobmsg_add_string(&b, "error", ubus_error_msg)) {
		fprintf(stderr, "blobmsg_add_string failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

	if (ubus_send_reply(&ubus_ctx, req, b.head)) {
		fprintf(stderr, "ubus_send_reply failure\n");
		rc = UBUS_STATUS_UNKNOWN_ERROR;
		goto clean;
	}

clean:
	blob_buf_free(&b);
	return rc;
}

static int valid_ring_recv(uint32_t recovery_time)
{
	switch (recovery_time) {
	case 500:
	case 200:
	case 30:
	case 10:
		break;
	default:
		return 1;
	}

	return 0;
}

static const char *alarm_to_str(ring_alarm_t alarm)
{
	switch (alarm) {
	case UBUS_ALARM_MANAGER_ROLE_OK:
		return "Manager Role Ok";
	case UBUS_ALARM_MANAGER_ROLE_FAIL:
		return "Manager Role Fail";
	case UBUS_ALARM_MULTIPLE_MANAGERS_OK:
		return "Multiple Managers Ok";
	case UBUS_ALARM_MULTIPLE_MANAGERS:
		return "Multiple Managers";
	case UBUS_ALARM_DOMAIN_MISMATCH_OK:
		return "Domain Mismatch OK";
	case UBUS_ALARM_DOMAIN_MISMATCH:
		return "Domain Mismatch";
	case UBUS_ALARM_RING_CLOSED:
		return "Ring Closed";
	case UBUS_ALARM_RING_OPEN:
		return "Ring Open";
	default:
		return "Unknown";
	}
}

int ubus_notify_alarm(ring_alarm_t alarm, int ring_nr)
{
	const char type[] = "mrp_alarm";
	struct blob_buf b = { 0 };
	int rc		  = 0;
	blob_buf_init(&b, 0);
	blobmsg_add_string(&b, "type", alarm_to_str(alarm));
	blobmsg_add_u32(&b, "code", alarm + (ring_nr << 4 & 0x0FF0));
	if ((rc = ubus_notify(&ubus_ctx, &ubus_mrp_object, type, b.head, -1))) {
		goto end;
	}
end:
	blob_buf_free(&b);
	return rc;
}
