// Copyright (c) 2020 Microchip Technology Inc. and its subsidiaries.
// SPDX-License-Identifier: (GPL-2.0)

#include <stdio.h>
#include <stdint.h>
#include <unistd.h>
#include <getopt.h>
#include <netlink/genl/genl.h>
#include <netlink/genl/ctrl.h>
#include <linux/types.h>
#include <net/if.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <stdbool.h>
#include <string.h>

#include "libnetlink.h"
#include "server_cmds.h"
#include "state_machine.h"
#include "list.h"
#include "netlink.h"
#include "cfm_netlink.h"
#include "print.h"

static struct rtnl_handle rth;
static ev_io netlink_watcher;
static LIST_HEAD(mrp_temp_instances);

static void mrp_tmp_destroy(struct mrp *mrp);

int CTL_addmrp(int br_index, int ring_nr, int pport, int sport, int ring_role,
	       uint16_t prio, uint8_t ring_recv, uint8_t react_on_link_change,
	       int in_role, uint16_t in_id, int iport, int in_mode,
	       uint8_t in_recv, int cfm_instance, int cfm_level, int cfm_mepid,
	       int cfm_peer_mepid, char *cfm_maid, char *cfm_dmac)
{
	return mrp_add(br_index, ring_nr, pport, sport, ring_role, prio,
		       ring_recv, react_on_link_change, in_role, in_id,
		       iport, in_mode, in_recv, cfm_instance, cfm_level,
		       cfm_mepid, cfm_peer_mepid, cfm_maid, cfm_dmac);
}

int CTL_delmrp(int br_index, int ring_nr)
{
	return mrp_del(br_index, ring_nr);
}

int CTL_getmrp(int *count, struct mrp_status *status)
{
	return mrp_get(count, status);
}

static struct mrp *mrp_tmp_find(char *ifname)
{
	struct mrp *mrp = NULL;

	if (!ifname) {
		return NULL;
	}

	list_for_each_entry (mrp, &mrp_temp_instances, list_tmp) {
		if (mrp->p_port && !strcmp(mrp->p_port->ifname, ifname)) {
			return mrp;
		} else if (mrp->s_port && !strcmp(mrp->s_port->ifname, ifname)) {
			return mrp;
		} else if (!strcmp(mrp->ifname, ifname)) {
			return mrp;
		}
	}

	return NULL;
}

static void mrp_tmp_find_all(char *ifname, int *count, struct mrp **instances)
{
	struct mrp *mrp = NULL;
	int i		= 0;

	if (!ifname) {
		return;
	}

	list_for_each_entry (mrp, &mrp_temp_instances, list_tmp) {
		if (!strcmp(mrp->p_port->ifname, ifname) || !strcmp(mrp->s_port->ifname, ifname) ||
		    !strcmp(mrp->ifname, ifname)) {
			instances[i] = mrp;
			i++;
		}
	}

	*count = i;
}

/* Creates an temporary MRP instance and initialize it for later use */
static struct mrp *mrp_tmp_create(char *br_ifname, uint32_t ring_nr, uint16_t in_id)
{
	struct mrp *mrp;

	mrp = malloc(sizeof(struct mrp));
	if (!mrp)
		return NULL;

	memset(mrp, 0x0, sizeof(struct mrp));

	pthread_mutex_init(&mrp->lock, NULL);

	strncpy(mrp->ifname, br_ifname, sizeof(mrp->ifname) - 1);

	mrp->p_port  = NULL;
	mrp->s_port  = NULL;
	mrp->i_port  = NULL;
	mrp->ring_nr = ring_nr;
	mrp->in_id   = in_id;

	mrp->ring_role	      = BR_MRP_RING_ROLE_MRC;
	mrp->in_role	      = BR_MRP_IN_ROLE_DISABLED;
	mrp->ring_transitions = 0;
	mrp->in_transitions   = 0;

	mrp->seq_id = 0;
	mrp->prio   = MRP_DEFAULT_PRIO;
	memset(mrp->domain, 0xFF, MRP_DOMAIN_UUID_LENGTH);

	mrp_update_recovery(mrp, MRP_RING_RECOVERY_500, MRP_IN_RECOVERY_500);

	mrp->blocked		  = 1;
	mrp->react_on_link_change = 1;

	list_add_tail(&mrp->list_tmp, &mrp_temp_instances);

	return mrp;
}

/* Initialize an tmp MRP port */
static int mrp_tmp_port_init(char *p_ifname, struct mrp *mrp, enum br_mrp_port_role_type role)
{
	struct mrp_port *port;

	port = malloc(sizeof(struct mrp_port));
	if (!port)
		return -ENOMEM;

	memset(port, 0x0, sizeof(struct mrp_port));

	port->mrp = mrp;
	strncpy(port->ifname, p_ifname, sizeof(port->ifname) - 1);
	port->role = role;

	if (role == BR_MRP_PORT_ROLE_PRIMARY)
		mrp->p_port = port;
	if (role == BR_MRP_PORT_ROLE_SECONDARY)
		mrp->s_port = port;
	if (role == BR_MRP_PORT_ROLE_INTER)
		mrp->i_port = port;

	return 0;
}

int mrp_add_tmp(char *br_ifname, uint32_t ring_nr, char *pport_ifname, char *sport_ifname, uint32_t ring_role,
		uint16_t prio, uint8_t ring_recv, uint8_t react_on_link_change, uint32_t in_role,
		uint16_t in_id, uint32_t iport, uint32_t in_mode, uint8_t in_recv, uint32_t cfm_instance,
		uint32_t cfm_level, uint32_t cfm_mepid, uint32_t cfm_peer_mepid, char *cfm_maid,
		char *cfm_dmac, uint8_t *domain, char *domain_name, struct mrp_params *prm)
{
	struct mrp *mrp;
	int err = 0;

	/* It is not possible to have tmp MRP instances with the same ports */
	if ((mrp = mrp_tmp_find(pport_ifname)) || (mrp = mrp_tmp_find(sport_ifname))) {
		return -EINVAL;
	}

	/* Create the mrp tmp instance */
	mrp = mrp_tmp_create(br_ifname, ring_nr, in_id);
	if (!mrp)
		return -ENOMEM;

	pthread_mutex_lock(&mrp->lock);

	mrp->prio = prio;
	mrp->ring_prio = prio;
	mrp_update_recovery(mrp, ring_recv, in_recv);
	mrp->react_on_link_change = react_on_link_change;
	mrp->in_mode = in_mode;
	mrp->ring_role = ring_role;

	if (domain)
		memcpy(mrp->domain, domain, MRP_DOMAIN_UUID_LENGTH);
	if (domain_name)
		strncpy((char *)mrp->domain_name, domain_name, MRP_DOMAIN_NAME_LENGTH - 1);
	if (prm) {
		mrp->ring_topo_conf_interval	  = prm->ring_topo_conf_interval;
		mrp->ring_topo_conf_max		  = prm->ring_topo_conf_max;
		mrp->ring_test_conf_short	  = prm->ring_test_conf_short;
		mrp->ring_test_conf_interval	  = prm->ring_test_conf_interval;
		mrp->ring_test_conf_max		  = prm->ring_test_conf_max;
		mrp->ring_test_conf_ext_max	  = prm->ring_test_conf_ext_max;
		mrp->ring_link_conf_interval_up	  = prm->ring_link_conf_interval_up;
		mrp->ring_link_conf_interval_down = prm->ring_link_conf_interval_down;
		mrp->ring_link_conf_max		  = prm->ring_link_conf_max;
		mrp->ring_recv			  = MRP_RING_RECOVERY_CUSTOM;
	}

	/* Initialize the ports */
	err = mrp_tmp_port_init(pport_ifname, mrp, BR_MRP_PORT_ROLE_PRIMARY);
	if (err < 0) {
		pthread_mutex_unlock(&mrp->lock);
		goto delete_mrp;
	}

	err = mrp_tmp_port_init(sport_ifname, mrp, BR_MRP_PORT_ROLE_SECONDARY);
	if (err < 0) {
		pthread_mutex_unlock(&mrp->lock);
		goto delete_port;
	}

	/* TODO: add interconnected port support
	err = mrp_tmp_port_init(iport_ifname, mrp, BR_MRP_PORT_ROLE_INTER);
	if (err < 0) {
		pthread_mutex_unlock(&mrp->lock);
		goto delete_ports;
	}
	*/

	pthread_mutex_unlock(&mrp->lock);

	if (err)
		goto clear;

	return 0;

clear:
	mrp_port_uninit(mrp->s_port);
	mrp->s_port = NULL;

delete_port:
	mrp_port_uninit(mrp->p_port);
	mrp->p_port = NULL;

delete_mrp:
	mrp_tmp_destroy(mrp);
	return err;
}

static struct mrp *create_tmp_mrp(struct mrp *mrp)
{
	struct mrp *tmp_mrp	= calloc(1, sizeof(*mrp));
	struct mrp_port *p_port = calloc(1, sizeof(struct mrp_port));
	struct mrp_port *s_port = calloc(1, sizeof(struct mrp_port));
	if (!tmp_mrp || !p_port || !s_port) {
		free(p_port);
		free(s_port);
		free(tmp_mrp);
		return NULL;
	}

	pthread_mutex_lock(&mrp->lock);
	memcpy(tmp_mrp, mrp, sizeof(struct mrp));
	// TODO: check case when pport and sport is swaped
	memcpy(p_port, mrp->p_port, sizeof(struct mrp_port));
	memcpy(s_port, mrp->s_port, sizeof(struct mrp_port));
	tmp_mrp->p_port = p_port;
	tmp_mrp->s_port = s_port;
	pthread_mutex_unlock(&mrp->lock);

	return tmp_mrp;
}

// TODO: port can't be null
// TODO: check mrp_add, mrp_del, because mrp instance can be in uninicialized state
static int add_mrp_tmp(struct mrp **tmp_mrp, struct mrp *mrp, char *ifname)
{
	struct mrp *p = NULL;
	if (!mrp)
		return -1;

	if (!(p = create_tmp_mrp(mrp)))
		return -1;

	list_add_tail(&p->list_tmp, &mrp_temp_instances);
	mrp_destroy(mrp->ifindex, mrp->ring_nr, true);
	*tmp_mrp = p;

	return 0;
}

static int update_mrp_tmp(struct mrp *mrp, char *ifname)
{
	struct mrp *tmp_mrp = NULL;
	if (!ifname) {
		pr_debug("no interface name");
		return -1;
	}

	if (!(tmp_mrp = mrp_tmp_find(ifname))) {
		if (add_mrp_tmp(&tmp_mrp, mrp, ifname)) {
			pr_debug("Failed to add mrp_tmp");
			return -1;
		}
	}

	if (!strcmp(tmp_mrp->p_port->ifname, ifname)) {
		tmp_mrp->p_port->ifindex = 0;
	} else if (!strcmp(tmp_mrp->s_port->ifname, ifname)) {
		tmp_mrp->s_port->ifindex = 0;
	} else if (!strncmp(tmp_mrp->ifname, ifname, strlen(tmp_mrp->ifname))) {
		tmp_mrp->ifindex = 0;
	}

	pr_debug("mrp_tmp updated");
	return 0;
}

static void mrp_tmp_destroy(struct mrp *mrp)
{
	free(mrp->p_port);
	free(mrp->s_port);
	list_del(&mrp->list_tmp);
	free(mrp);
}

static int reset_mrp_tmp(char *ifname, struct ifinfomsg *ifi)
{
	int rc = 0, count = 0;
	struct mrp *mrps[MAX_MRP_INSTANCES] = { 0 };

	// Should find one with port interface, multiple when bridge
	mrp_tmp_find_all(ifname, &count, mrps);
	if (count == 0) {
		goto end;
	}

	for (int i = 0; i < count; i++) {
		struct mrp_params *prm = NULL;
		if (!strcmp(mrps[i]->p_port->ifname, ifname)) {
			mrps[i]->p_port->ifindex = if_nametoindex(ifname);
		} else if (!strcmp(mrps[i]->s_port->ifname, ifname)) {
			mrps[i]->s_port->ifindex = if_nametoindex(ifname);
		} else if (!strcmp(mrps[i]->ifname, ifname) &&
			   (ifi->ifi_flags & IFF_RUNNING)) { // TODO: check br0.1 case
			pr_debug("bridge br0 is running again");
			mrps[i]->ifindex = if_nametoindex(ifname);
		}

		if (mrps[i]->ring_recv == MRP_RING_RECOVERY_CUSTOM) {
			if (!(prm = calloc(1, sizeof(*prm)))) {
				rc = -1;
				goto end;
			}

			prm->ring_topo_conf_interval	  = mrps[i]->ring_topo_conf_interval;
			prm->ring_topo_conf_max		  = mrps[i]->ring_topo_conf_max;
			prm->ring_test_conf_short	  = mrps[i]->ring_test_conf_short;
			prm->ring_test_conf_interval	  = mrps[i]->ring_test_conf_interval;
			prm->ring_test_conf_max		  = mrps[i]->ring_test_conf_max;
			prm->ring_test_conf_ext_max	  = mrps[i]->ring_test_conf_ext_max;
			prm->ring_link_conf_interval_up	  = mrps[i]->ring_link_conf_interval_up;
			prm->ring_link_conf_interval_down = mrps[i]->ring_link_conf_interval_down;
			prm->ring_link_conf_max		  = mrps[i]->ring_link_conf_max;
		}

		if (mrps[i]->p_port->ifindex && mrps[i]->s_port->ifindex && mrps[i]->ifindex) {
			// TODO: add interconnected port recovery support
			if (!(rc = mrp_add_v2(mrps[i]->ifindex, mrps[i]->ring_nr, mrps[i]->p_port->ifindex,
					      mrps[i]->s_port->ifindex, mrps[i]->ring_role, mrps[i]->prio,
					      mrps[i]->ring_recv, mrps[i]->react_on_link_change,
					      mrps[i]->in_role, mrps[i]->in_id, 0, mrps[i]->in_mode,
					      mrps[i]->in_recv, 0, 0, 0, 0, NULL, NULL, mrps[i]->domain,
					      (char *)mrps[i]->domain_name, prm))) {
				mrp_tmp_destroy(mrps[i]);
				pr_debug("MRP instance %d", mrps[i]->ring_nr);
			}
		}

		free(prm);
	}

end:
	return rc;
}

static int netlink_listen(struct rtnl_ctrl_data *who, struct nlmsghdr *n,
			  void *arg)
{
	struct rtattr *infotb[IFLA_BRIDGE_CFM_MEP_STATUS_MAX + 1];
	struct rtattr *aftb[IFLA_BRIDGE_MAX + 1];
	struct ifinfomsg *ifi = NLMSG_DATA(n);
	struct rtattr * tb[IFLA_MAX + 1];
	struct mrp_port *port = NULL;
	int len = n->nlmsg_len;
	int af_family;
	int rem, instance;
	struct rtattr *i, *list;
	
	if (n->nlmsg_type == NLMSG_DONE)
		return 0;

	len -= NLMSG_LENGTH(sizeof(*ifi));
	if (len < 0)
		return -1;

	af_family = ifi->ifi_family;

	port = mrp_get_port(ifi->ifi_index);

	if (af_family != AF_BRIDGE && af_family != AF_UNSPEC)
		return 0;

	if (n->nlmsg_type != RTM_NEWLINK && n->nlmsg_type != RTM_DELLINK)
		return 0;

	parse_rtattr_flags(tb, IFLA_MAX, IFLA_RTA(ifi), len, NLA_F_NESTED);

	if (tb[IFLA_IFNAME] == NULL) {
		pr_err("BUG: nil ifname");
		return -1;
	}

	char *ifname = (char *)RTA_DATA(tb[IFLA_IFNAME]);

	if (tb[IFLA_PROTINFO]) {
		// Prioritise notifications about ring state
		struct rtattr *aftc[IFLA_BRPORT_MAX + 1];

		parse_rtattr_nested(aftc, IFLA_BRPORT_MAX, tb[IFLA_PROTINFO]);

		if (port) {
			if (aftc[IFLA_BRPORT_MRP_RING_OPEN]) {
				mrp_port_ring_open(port, rta_getattr_u8(aftc[IFLA_BRPORT_MRP_RING_OPEN]));
			}

			if (aftc[IFLA_BRPORT_MRP_IN_OPEN]) {
				mrp_port_in_open(port, rta_getattr_u8(aftc[IFLA_BRPORT_MRP_IN_OPEN]));
			}
		} else {
			pr_debug("port was not found");
		}
	}

	if (n->nlmsg_type == RTM_DELLINK) {
		pr_debug("RTM_DELLINK detected on interface %s - pid %d", ifname, n->nlmsg_pid);
		struct mrp *mrp = port ? port->mrp : NULL;
		update_mrp_tmp(mrp, ifname);

		// check bridge
		// TODO: fix should be added to linux kernel this should be temporary for userspace
		// netlink seems to break when bridge vlan interface is deleted
		// more investigation needed
		struct mrp *instances[MAX_MRP_INSTANCES] = { 0 };
		int count				 = 0;
		mrp_find_all(ifname, &count, instances);
		for (int i = 0; i < count; i++) {
			pr_debug("RTM_DELLINK detected on mrp bridge %s ring_nr %d", ifname,
				 instances[i]->ring_nr);
			update_mrp_tmp(instances[i], ifname);
		}
	}

	if (n->nlmsg_type == RTM_NEWLINK && (ifi->ifi_flags & IFF_UP)) {
		pr_debug("RTM_NEWLINK detected on interface %s is ready - pid %u ", ifname, n->nlmsg_pid);
		reset_mrp_tmp(ifname, ifi);
	}

	if (tb[IFLA_ADDRESS]) {
		mrp_mac_change(ifi->ifi_index,
			       (__u8*)RTA_DATA(tb[IFLA_ADDRESS]));
	}

	if (tb[IFLA_AF_SPEC]) {
		parse_rtattr_flags(aftb, IFLA_BRIDGE_MAX, RTA_DATA(tb[IFLA_AF_SPEC]),
				   RTA_PAYLOAD(tb[IFLA_AF_SPEC]), NLA_F_NESTED);
		if (!aftb[IFLA_BRIDGE_CFM])
			goto mrp_process;

		list = aftb[IFLA_BRIDGE_CFM];
		rem = RTA_PAYLOAD(list);

		instance = 0xFFFFFFFF;
		for (i = RTA_DATA(list); RTA_OK(i, rem); i = RTA_NEXT(i, rem)) {
			if (i->rta_type != (IFLA_BRIDGE_CFM_CC_PEER_STATUS_INFO | NLA_F_NESTED))
				continue;

			parse_rtattr_flags(infotb, IFLA_BRIDGE_CFM_CC_PEER_STATUS_MAX, RTA_DATA(i), RTA_PAYLOAD(i), NLA_F_NESTED);
			if (!infotb[IFLA_BRIDGE_CFM_CC_PEER_STATUS_INSTANCE])
				continue;

			if (instance != rta_getattr_u32(infotb[IFLA_BRIDGE_CFM_CC_PEER_STATUS_INSTANCE])) {
				instance = rta_getattr_u32(infotb[IFLA_BRIDGE_CFM_CC_PEER_STATUS_INSTANCE]);
			}

			mrp_cfm_link_change(ifi->ifi_index, rta_getattr_u32(infotb[IFLA_BRIDGE_CFM_CC_PEER_STATUS_PEER_MEPID]),
					    rta_getattr_u32(infotb[IFLA_BRIDGE_CFM_CC_PEER_STATUS_CCM_DEFECT]));
		}
	}

mrp_process:
	if (!port)
		return 0;

	if (tb[IFLA_OPERSTATE]) {
		__u8 state = *(__u8*)RTA_DATA(tb[IFLA_OPERSTATE]);
		if (port->operstate != state) {
			port->operstate = state;

			switch (state) {
			case IF_OPER_UNKNOWN:
			case IF_OPER_NOTPRESENT:
			case IF_OPER_DOWN:
			case IF_OPER_LOWERLAYERDOWN:
			case IF_OPER_TESTING:
			case IF_OPER_DORMANT:
				pr_debug("NETLINK_LISTEN -> port %s is down", port->ifname);
				mrp_port_link_change(port, false);
				break;

			case IF_OPER_UP:
				pr_debug("NETLINK_LISTEN -> port %s is up", port->ifname);
				mrp_port_link_change(port, true);
				break;
			}
		}
	}

	if (!tb[IFLA_MASTER]) {
		mrp_destroy(port->mrp->ifindex, port->mrp->ring_nr, false);
	}

	return 0;
}

static void netlink_rcv(EV_P_ ev_io *w, int revents)
{
	rtnl_listen(&rth, netlink_listen, stdout);
}

static int netlink_init(void)
{
	int err;

	err = rtnl_open(&rth, RTMGRP_LINK);
	if (err)
		return err;

	fcntl(rth.fd, F_SETFL, O_NONBLOCK);

	ev_io_init(&netlink_watcher, netlink_rcv, rth.fd, EV_READ);
	ev_io_start(EV_DEFAULT, &netlink_watcher);

	return 0;
}

static void netlink_uninit(void)
{
	ev_io_stop(EV_DEFAULT, &netlink_watcher);
	rtnl_close(&rth);
}

int CTL_init(void)
{
	if (netlink_init()) {
		pr_err("netlink init failed!");
		return -1;
	}

	if (if_init()) {
		pr_err("if init failed");
		return -1;
	}

	if (mrp_netlink_init()) {
		pr_err("mrp netlink init failed");
		return -1;
	}

	if (cfm_offload_init()) {
		pr_err("cfm offload init failed");
		return -1;
	}

	return 0;
}

void CTL_cleanup(void)
{
	mrp_uninit();
	mrp_netlink_uninit();
	netlink_uninit();
}
