#!/usr/bin/env lua
require "ubus"
require "uloop"
local RECHECK_TIME = 5000
local json = require("luci.jsonc")
local uci = require("uci")
local conn = ubus.connect()
if not conn then error("Failed to connect to ubus") end
uloop.init()
local last_matches = { }
local last_warning_time = {}

local function violation_action(port, cursor)
	if cursor:get("macfilter", port, "intruder_action") ~= "block_port" or
		cursor:get("macfilter", port, "enabled") ~= "1" then
		return
	end
	if last_warning_time[port] or 0 + 15000 < os.time() then
		conn:call("log", "write_ext", {
			event = "Unauthorized MAC address detected on port "..port..". Disabling port.",
			sender = "Macfilter",
			table = 2,
			priority = 4,
			write_db = 1
		})
	end
	last_warning_time[port] = os.time()
	cursor:set("tswconfig", port, "enable", "0")
	cursor:commit("tswconfig")
	conn:call("file", "exec", {command = "/etc/init.d/tswconfig", params = {"reload"}})
end

local function poll_tc()
	local res = conn:call("file", "exec", {command = "/sbin/tc", params = {"-j", "-s", "filter", "show", "dev", "eth0", "ingress"}})

	local function parse_rule(obj)
		if obj.protocol ~= "all" or obj.kind ~= "flower" then return end
		local opts = obj.options
		if not opts or not opts.actions then return end
		local act = opts.actions[1]
		if not act or not act.control_action or act.control_action.type ~= "drop" then return end
		local dev = obj.options.indev
		local matched_bytes = tonumber(act.stats.hw_bytes) or 0
		if not last_matches[dev] then last_matches[dev] = matched_bytes end
		if matched_bytes > last_matches[dev] then violation_action(dev, uci.cursor()) end
		last_matches[dev] = matched_bytes
	end

	if res.code ~= 0 then return end

	local parsed = json.parse(res.stdout) or {}
	for _, obj in ipairs(parsed) do
		parse_rule(obj)
	end
end

local poll_timer
poll_timer = uloop.timer(function()
	poll_timer:set(RECHECK_TIME)
	poll_tc()
end)
poll_timer:set(500)
uloop.run()
