#ifndef LIBMRP_H
#define LIBMRP_H

#include <libubus.h>
#include <stdint.h>
#include <sys/types.h>
#include <net/if.h>

#define LMRP_DOMAIN_MAX_LEN 16
#define LMRP_DOMAIN_NAME_MAX_LEN 240

// TODO: 0x8010 – 0x87EFReserved for “Multiple MRP rings”
// ExtChannelErrorType = 0x800X + (MRP_Instance × 0x10)


// TODO: interconnection roles
// TODO: Manager Automanager roles
// TODO: enums to string

// NOTE: first dev. iteration will be orentaded for MRC role

enum lmrp_alarm_type {
	LMRP_ALARM_MANAGER_ROLE_OK	= 0x0000,
	LMRP_ALARM_MANAGER_ROLE_FAIL	= 0x8000,
	LMRP_ALARM_RING_CLOSED		= 0x0001,
	LMRP_ALARM_RING_OPEN		= 0x8001,
	LMRP_ALARM_MULTIPLE_MANAGERS_OK = 0x0003,
	LMRP_ALARM_MULTIPLE_MANAGERS	= 0x8003,
	LMRP_ALARM_DOMAIN_MISMATCH_OK	= 0x0004,
	LMRP_ALARM_DOMAIN_MISMATCH	= 0x8004,
};

enum lmrp_mrm_state_type {
	/* Awaiting Connection State 1 */
	LMRP_MRM_STATE_AC_STAT1 = 0x0,
	/* Primary Ring port with Link Up */
	LMRP_MRM_STATE_PRM_UP = 0x1,
	/* Check Ring, Ring Open State */
	LMRP_MRM_STATE_CHK_RO = 0x2,
	/* Check Ring, Ring Closed State */
	LMRP_MRM_STATE_CHK_RC = 0x3,
};

enum lmrp_mrc_state_type {
	/* Awaiting Connection State 1 */
	LMRP_MRC_STATE_AC_STAT1 = 0x0,
	/* Data Exchange Idle state */
	LMRP_MRC_STATE_DE_IDLE = 0x1,
	/* Pass Through */
	LMRP_MRC_STATE_PT = 0x2,
	/* Data Exchange */
	LMRP_MRC_STATE_DE = 0x3,
	/* Pass Through Idle state */
	LMRP_MRC_STATE_PT_IDLE = 0x4,
};

enum lmrp_recv_type {
	LMRP_RING_RECV_500,
	LMRP_RING_RECV_200,
	LMRP_RING_RECV_30,
	LMRP_RING_RECV_10,
};

enum lmrp_ring_role_type {
	LMRP_ROLE_MRC = 0x0001,
	LMRP_ROLE_MRM,
	LMRP_ROLE_MRA,
};

enum lmrp_ring_state_type {
	LMRP_OPEN = 0x0001,
	LMRP_CLOSED,
};

enum lmrp_port_state_type {
	LMRP_PORT_STATE_DISABLED,
	LMRP_PORT_STATE_BLOCKED,
	LMRP_PORT_STATE_FORWARDING,
	LMRP_PORT_STATE_NOT_CONNECTED,
};

// TODO: errors not so informative for now
typedef enum {
	LMRP_OK,
	LMRP_UNSUPPORTED_ERR = __UBUS_STATUS_LAST,
	LMRP_PARSING_FAILED,
	LMRP_NOT_FOUND_ERR,
	LMRP_UBUS_ERR,
	__LMRP_LAST,
} lmrp_err_t;

typedef struct {
	uint16_t ring_state;
	uint16_t pport_state;
	uint16_t sport_state;
} lmrp_info_t;

typedef struct {
	uint32_t topchgt;
	uint32_t topnrmax;
	uint32_t tstshortt;
	uint32_t tstdefaultt;
	uint32_t tstnrmax;
	uint32_t tstextnrma;
	uint32_t lnkupt;
	uint32_t lnkdownt;
	uint32_t lnknrmax;
} lmrp_prm_t;

typedef struct {
	uint8_t instance_nr; // MRP_Instance
	uint8_t domain[LMRP_DOMAIN_MAX_LEN]; // Default: filled with 0xFF
	uint8_t domain_name[LMRP_DOMAIN_NAME_MAX_LEN];
	uint8_t domain_name_length;
	uint8_t bridge[IF_NAMESIZE];
	uint8_t pport[IF_NAMESIZE];
	uint8_t sport[IF_NAMESIZE];
	uint16_t prio;
	uint16_t version; // Default: 0x0001
	enum lmrp_recv_type ring_recv;
	enum lmrp_ring_role_type role;
	lmrp_info_t info;
	lmrp_prm_t *prm;
} lmrp_t;

typedef struct {
	lmrp_t *arr;
	size_t cnt;
} lmrp_arr_t;

typedef struct {
	uint8_t instance_nr;
	enum lmrp_alarm_type type;
} lmrp_alarm_t;

/**
 * @brief add mrp instance 
 * @note you can add custom parameter values by allocating lmrp_prm_t pointer
 * @note by default parameters will be set by EN 62439-2 profiles
 * @note prm time is in micro seconds
 * @param ubus context of the ubus
 * @param lmrp reference to lmrp buffer (must be allocated)
 * @param update_config update config file
 * @return error type
 */
lmrp_err_t lmrp_add(struct ubus_context *ubus, lmrp_t *lmrp, bool update_config);

/**
 * @brief returns all mrp instances
 * @param ubus context of the ubus
 * @param lmrp reference to lmrp_arr buffer (must be allocated and initialized)
 * @return error type
 */
lmrp_err_t lmrp_all(struct ubus_context *ubus, lmrp_arr_t *lmrp_arr);

/**
 * @brief deletes mrp instances
 * @param ubus context of the ubus
 * @param bridge network interface name of a bridge
 * @param instance_nr id of mrp instance
 * @param update_config update config file
 * @return error type
 */
lmrp_err_t lmrp_del(struct ubus_context *ubus, uint8_t *bridge, uint8_t instance_nr, bool update_config);

/**
 * @brief get mrp instance
 * @param ubus context of the ubus
 * @param bridge network interface name of a bridge
 * @param instance_nr id of mrp instance
 * @return error type
 */
lmrp_err_t lmrp_get(struct ubus_context *ubus, lmrp_t *lmrp, uint8_t *bridge, uint8_t instance_nr);

/**
 * @brief convert error enum to string literal
 * @param error type
 * @return error in string form
 */
const char *lmrp_strerror(lmrp_err_t err);

/**
 * @brief function subscribes mrp ubus object for ring state diagnostics
 * 	  events are generated only from MRM and MRA roles
 *
 * @param ubus context of the ubus
 * @param mrp_sub ubus subscriber
 * @param cb user defined callback
 * @param rcb user defined ubus remove handler callback 
 * @return error type
 */
lmrp_err_t lmrp_subscribe(struct ubus_context *ubus, struct ubus_subscriber *mrp_sub, ubus_handler_t cb,
			  ubus_remove_handler_t rcb);

/**
 * @brief function parses message blob from ubus alarm events
 *        use this in ubus_handler_t callback
 * @param alarm type and instance number (must be allocated)
 * @param msg blob attribute of a message received from ubus
 * @return errot type
 */

lmrp_err_t lmrp_parse_alarm_msg(lmrp_alarm_t *alarm, struct blob_attr *msg);

/**
 * @brief update mrp instance
 * @description
 * This wrapper simply reinicializes mrp instance with new settigs
 * deletes old instance and creates new one
 * bridge and instance_nr must be initialized for finding already
 * created mrp instance
 * @param ubus context of the ubus
 * @param lmrp reference to lmrp buffer (must be allocated and initialized)
 * @param update_config update config file
 * @return error type
 */
lmrp_err_t lmrp_upd(struct ubus_context *ubus, lmrp_t *lmrp, bool update_config);

/**
 * @brief MRM and MRC parameters described in EN 62439-2
 * @description 
 *
 * all parameters are constants in five consitent parameter profiles
 * described in EN 62439-2 9.3 section
 * this function return parameters via reference for the given instance
 * parameters representing timestamp are initialized in microseconds
 * @note profiles were defined according to microchip MRP implementation
 * @note new feature lets add custom parameter values see lmrp_add
 * @note prm time is in micro seconds
 * @param ubus context of the ubus
 * @param prm instance parameters (must be allocated)
 * @param bridge network interface name of a bridge
 * @param instance_nr id of mrp instance
 */
lmrp_err_t lmrp_prm(struct ubus_context *ubus, lmrp_prm_t *prm, uint8_t *bridge, uint8_t instance_nr);

/**
 * @brief free array of mrp instances
 * @param lmrp_arr struct that holds instances and count
 */
void lmrp_arr_free(lmrp_arr_t *lmrp_arr);

/**
 * @brief convert role enum to string
 * @param role role enumerator
 */
const char *lmrp_role_to_str(enum lmrp_ring_role_type role);

/**
 * @brief convert string to role enum
 * @param role role string
 */
enum lmrp_ring_role_type lmrp_str_to_role(char *role);

/**
 * @brief convert int to recv enum
 * @param ring_recv recovery time in ms
 */
enum lmrp_recv_type lmrp_int_to_recv_time(int ring_recv);

/**
 * @brief convert recv enum to int
 * @param ring_recv enumerator
 */
int lmrp_ring_recv_to_int(enum lmrp_recv_type ring_recv);

#endif // LIBMRP_H
