#!/usr/bin/env lua
require "ubus"
require "uloop"
local util = require("vuci.util")
local fs = require("nixio.fs")
local uci_lib = require("uci")
local uci = uci_lib.cursor()

local CONFIG_DIR = "/tmp/dot1x_client/"

local conn = ubus.connect()
uloop.init()
if not conn then
	error("Failed to connect to ubus")
end

local dot1x_clients = { }

local function in_table(table, value, key)
	for k, v in pairs(table) do
		if key and k == value then return true end
		if not key and v == value then return true end
	end
	return false
end

local function reload_port_wpa_supplicant(port)
	port = string.lower(port)
	if dot1x_clients[port].enabled ~= "1" or dot1x_clients[port].role ~= "client" then return end
	local obj_name = "wpa_supplicant."..port
	if not in_table(conn:objects(), obj_name) then return end
	local timer = uloop.timer(function()
		print("Reloading wpa supplicant for port "..port)
		conn:call(obj_name, "reload", {})
	end)
	timer:set(1000) -- give time for the other end to prepare 802.1x
end

local function parse_config(client)
	if client.enabled ~= "1" or client.role ~= "client" then return false end
	local config = {key_mgmt="IEEE8021X", eapol_flags="0"}
	local function add_string(cfg, option, value, path)
		if not value then return end
		if path and not fs.access(value) then return end
		cfg[option] = string.format('"%s"', value)
	end

	config.eap = string.upper(client.auth_type)
	local features = {
		identity           = {cfg_name = "identity", used_in = {"md5", "tls", "pwd", "ttls", "peap"}},
		anonymous_identity = {cfg_name = "anonymous_identity", used_in = {"ttls", "peap"}},
		ca_cert            = {cfg_name = "ca_cert", used_in = {"tls", "ttls", "peap"}, file = true},
		private_key        = {cfg_name = "private_key", used_in = {"tls"}},
		private_key_passwd = {cfg_name = "private_key_pass", used_in = {"tls"}},
		client_cert        = {cfg_name = "client_cert", used_in = {"tls"}},
		password           = {cfg_name = "password", used_in = {"md5", "pwd", "ttls", "peap"}},
	}

	for key, info in pairs(features) do
		if in_table(info.used_in, client.auth_type) then
			add_string(config, key, client[info.cfg_name], info.file)
		end
	end

	local ttls_phase2_map = {
		["MSCHAPV2"]      = '"autheap=MSCHAPV2"',
		["MD5"]           = '"autheap=MD5"',
		["GTC"]           = '"autheap=GTC"',
		["MSCHAPV2NOEAP"] = '"auth=MSCHAPV2"',
		["PAP"]           = '"auth=PAP"',
		["MSCHAP"]        = '"auth=MSCHAP"',
		["CHAP"]          = '"auth=CHAP"'
	}

	-- too custom to add through the features table
	if client.auth_type == "ttls" then
		config.phase2 = ttls_phase2_map[client.inner_authentication]
	elseif client.auth_type == "peap" then
		config.phase2 = string.format('"auth=%s"', client.inner_authentication)
		config.phase1 = string.format('"peapver=%s"', client.peap_version)
	end

	return config
end

local old_config = {}

local function write_config_to_file(file, config)
	local config_string = "network={\n"
	for key, value in pairs(config) do
		if not string.match(key, "^_priv_") then
			config_string = config_string..string.format("	%s=%s\n", key, value)
		end
	end
	config_string = config_string.."}\n"
	fs.writefile(file, config_string)
end

local ERROR_CODES = {
	SUCCESS = 4
}
local statuses = {}

local function generate_configuration()
	statuses = {}
	os.execute("rm -fr "..CONFIG_DIR)
	fs.mkdirr(CONFIG_DIR)
	local configs = {}
	local wpa_supplicants = {}
	uci:foreach("dot1x", "port", function(client)
		local config = parse_config(client)
		if not config then return end
		local name = client[".name"]
		configs[name] = config
		local old_cfg = old_config[name]
		local cfg_path = CONFIG_DIR..name
		write_config_to_file(cfg_path, config)
		wpa_supplicants[name] = { driver = "wired", iface = name, config = cfg_path, bridge = "br0", reload = not old_cfg or not util.deep_compare(old_cfg, config) }
		statuses[name] = {code=ERROR_CODES.SUCCESS}
	end)

	for phy, info in pairs(wpa_supplicants) do
		if not info or info.reload then
			print("Removing wpa_supplicant for phy: "..phy)
			conn:call("wpa_supplicant", "config_remove", {iface = phy})
		end
		if info and info.reload then
			info.reload = nil
			print("Adding wpa_supplicant for phy: "..phy)
			conn:call("wpa_supplicant", "config_add", info)
		end
	end
	old_config = configs
end

local function refresh_tables()
	uci:foreach("dot1x", "port", function(client)
		dot1x_clients[client[".name"]] = client
	end)
end

local method = {
	dot1x_client = {
		reload = {
			function()
				uci = uci_lib.cursor()
				dot1x_clients = {}
				refresh_tables()
				generate_configuration()
			end, { }
		},
		status = {
			function(req)
				conn:reply(req, statuses)
			end, {}
		},
		kill_clients = {
			function()
				for phy, _ in pairs(statuses) do
					print("Removing wpa_supplicant for phy: "..phy)
					conn:call("wpa_supplicant", "config_remove", {iface = phy})
				end
				old_config = {}
			end, {}
		}
	}
}

local timer
local port_states = {}
local function port_events()
	-- polling for port changes because no port_eventsd and this is lua so no inotify afaik
	for port, _ in pairs(dot1x_clients) do
		local operstate = fs.readfile("/sys/class/net/"..port.."/operstate")
		local old_state = port_states[port]
		port_states[port] = not (string.match(operstate, ".*down.*") and true or false)
		if old_state ~= nil and port_states[port] and old_state ~= port_states[port] then
			reload_port_wpa_supplicant(port)
		end
	end
	timer:set(3000)
end

timer = uloop.timer(port_events)
conn:add(method)
refresh_tables()
generate_configuration()
port_events()
uloop.run()

