#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/hwmon.h>
#include <linux/platform_device.h>

#define DRV_NAME "fan-rpm"

#define WAIT_MS	     3000
#define IRQ_CNT_THR  100
#define DUTY_CNT_THR 12

struct fan_rpm_data {
	struct device *dev;
	struct timer_list timer;
	const char *label;
	struct device *hwmon_dev;
	uint8_t alarm;
	int min_rpm;
	int cur_rpm;
	int new_rpm;
	int n_sens;
	int duty_max; // As percentage
	int duty_min; // As percentage
	int irq;

	uint8_t irq_disabled;
	int irq_counter;
	int duty_cycles;
	int64_t time_total_ns; // Collected time intervals with OK duty cycles
	ktime_t h1_time; // Prev interrupt handling time
	ktime_t h2_time; // prev-prev interrupt handling time
};

static ssize_t rpm_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fan_rpm_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%d\n", data->cur_rpm);
}

static ssize_t alarm_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fan_rpm_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%d\n", data->alarm);
}

static DEVICE_ATTR_RO(rpm);
static DEVICE_ATTR_RO(alarm);

static struct attribute *fan_rpm_attr[] = {
	&dev_attr_rpm.attr,
	&dev_attr_alarm.attr,
	NULL,
};

static struct attribute_group fan_rpm_group = {
	.attrs = fan_rpm_attr,
};

static const struct attribute_group *fan_rpm_groups[] = { &fan_rpm_group, NULL };

static void read_rpm_cb(struct timer_list *t)
{
	struct fan_rpm_data *data = container_of(t, struct fan_rpm_data, timer);

	dev_dbg(data->hwmon_dev, "cur-rpm:%d\n", data->cur_rpm);

	/* Check if alarm state changed */
	if ((data->new_rpm < data->min_rpm && !data->alarm) ||
	    (data->new_rpm >= data->min_rpm && data->alarm)) {
		data->alarm = data->new_rpm < data->min_rpm;

		/* This may be spamed if fans are alternating over
		 * the threshold of spinning. Thus "dev_dbg" */
		dev_dbg(data->hwmon_dev, "Alarm state changed to %d\n", data->alarm);
		sysfs_notify(&data->hwmon_dev->kobj, NULL, "alarm");
		kobject_uevent(&data->hwmon_dev->kobj, KOBJ_CHANGE);
	}

	if (data->irq_disabled) {
		dev_dbg(data->hwmon_dev, "Enabling interrupt:%d\n", data->irq);
		enable_irq(data->irq);
		data->irq_disabled = 0;
	}

	data->cur_rpm	  = data->new_rpm;
	data->new_rpm	  = 0;
	data->irq_counter = 0;
	data->duty_cycles = 0;

	mod_timer(t, jiffies + msecs_to_jiffies(WAIT_MS));
}

static void stop_counting(struct fan_rpm_data *data)
{
	data->irq_disabled = 1;
	disable_irq_nosync(data->irq);
	data->h1_time	    = 0;
	data->h2_time	    = 0;
	data->time_total_ns = 0;
}

/* Checks if duty cycle is OK by 3 sequential fronts */
static int64_t duty_filter(struct fan_rpm_data *fdata, ktime_t current_time)
{
	int64_t last_interv_ns;
	int64_t period_ns;
	int duty_cycle;

	/* We do not yet know timings of 3 sequential fronts */
	if (!fdata->h2_time || !fdata->h1_time) {
		fdata->h2_time = fdata->h1_time;
		fdata->h1_time = current_time;
		return -1;
	}

	last_interv_ns = ktime_to_ns(ktime_sub(current_time, fdata->h1_time));
	period_ns      = ktime_to_ns(ktime_sub(current_time, fdata->h2_time));

	/* Cycle timings */
	fdata->h2_time = fdata->h1_time;
	fdata->h1_time = current_time;

	if (period_ns > 0) {
		duty_cycle = div_s64((last_interv_ns * 100), period_ns);

		/* If duty cycle is invalid - reset timings */
		if (duty_cycle < fdata->duty_min || duty_cycle > fdata->duty_max) {
			dev_dbg(fdata->hwmon_dev, "Duty cycle %d%%", duty_cycle);
			fdata->h1_time = 0;
			fdata->h2_time = 0;
			return -1;
		} else {
			return period_ns;
		}
	}

	return -1;
}

static irqreturn_t rpm_irq_handler(int irq, void *data)
{
	struct fan_rpm_data *fdata = data;
	int32_t reminder;
	int64_t interval;
	ktime_t cur = ktime_get();

	fdata->irq_counter++;

	/* Prevent interrupt spam for whole WAIT_MS */
	if (fdata->irq_counter > IRQ_CNT_THR) {
		dev_dbg(fdata->hwmon_dev, "Preventing RPM interrupt flood\n");
		stop_counting(fdata);
	}

	interval = duty_filter(fdata, cur);
	/* Skip rising fronts and interrupts with invalid duty cycles (interval == 0) */
	if (interval < 0) {
		return IRQ_HANDLED;
	}

	fdata->time_total_ns += interval;
	fdata->duty_cycles++;
	if (fdata->duty_cycles == DUTY_CNT_THR) {
		fdata->new_rpm = div_s64_rem(NSEC_PER_SEC, fdata->time_total_ns, &reminder) *
				 fdata->duty_cycles * (60 / fdata->n_sens);
		/* Do not discard remaider part */
		fdata->new_rpm += div_s64((int64_t)reminder * fdata->duty_cycles * (60 / fdata->n_sens),
					  fdata->time_total_ns);

		dev_dbg(fdata->hwmon_dev, "Disabling interrupt %d\n", irq);
		stop_counting(fdata);
	}

	return IRQ_HANDLED;
}

static struct of_device_id fan_rpm_dt_ids[] = {
	{
		.compatible = DRV_NAME,
	},
	{},
};

MODULE_DEVICE_TABLE(of, fan_rpm_dt_ids);

static int fan_rpm_probe(struct platform_device *pdev)
{
	struct device *dev;
	struct device_node *node;
	struct fan_rpm_data *data;
	int size, ret;
	const __be32 *p;
	const char *label;

	dev  = &pdev->dev;
	node = dev->of_node;

	dev_info(dev, "Probing");

	data = devm_kzalloc(dev, sizeof(struct fan_rpm_data), GFP_KERNEL);
	if (!data)
		return -ENOMEM;

	p = of_get_property(node, "min-rpm", &size);
	if (!p) {
		dev_err(dev, "'%s' property is missing", "min-rpm");
		return -ENODEV;
	}

	data->min_rpm = be32_to_cpup(p);

	p = of_get_property(node, "n-sens", &size);
	if (!p) {
		dev_err(dev, "'%s' property is missing", "n-sens");
		return -ENODEV;
	}

	data->n_sens = be32_to_cpup(p);

	label = of_get_property(node, "label", NULL);
	if (!label) {
		dev_err(dev, "'%s' property is missing", "label");
		return -ENODEV;
	}

	data->label = label;

	p = of_get_property(node, "duty", &size);
	if (!p || size != 2 * sizeof(*p)) {
		dev_err(dev, "'%s' property is missing", "duty");
		return -ENODEV;
	}

	data->duty_min = be32_to_cpup(p);
	data->duty_max = be32_to_cpup(p + 1);

	/* Make this driver part of hwmon class. */
	data->hwmon_dev = devm_hwmon_device_register_with_groups(dev, label, data, fan_rpm_groups);
	if (IS_ERR(data->hwmon_dev)) {
		return PTR_ERR(data->hwmon_dev);
	}

	data->irq = platform_get_irq(pdev, 0);
	if (data->irq < 0) {
		dev_err(dev, "Failed to obtain IRQ number: %d\n", data->irq);
		return data->irq;
	}

	ret = devm_request_irq(dev, data->irq, rpm_irq_handler, IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING,
			       data->label, data);
	if (ret < 0) {
		dev_err(dev, "Failed to request IRQ: %d\n", ret);
		return ret;
	}

	platform_set_drvdata(pdev, data);

	timer_setup(&data->timer, read_rpm_cb, 0);
	mod_timer(&data->timer, jiffies + msecs_to_jiffies(WAIT_MS));

	return 0;
}

static int fan_rpm_remove(struct platform_device *pdev)
{
	struct fan_rpm_data *data = dev_get_drvdata(&pdev->dev);

	del_timer_sync(&data->timer);

	/* Not sure if these are strictly needed. But let's be safe */
	disable_irq(data->irq);
	synchronize_irq(data->irq);

	return 0;
}

static struct platform_driver fan_rpm_driver = {
	.probe = fan_rpm_probe,
	.remove= fan_rpm_remove,
	.driver = {
		.name = DRV_NAME,
		.of_match_table = fan_rpm_dt_ids,
	},
};

static int __init fan_rpm_init(void)
{
	if (platform_driver_register(&fan_rpm_driver)) {
		pr_err("Cannot load driver:%s\n", fan_rpm_driver.driver.name);
		return -1;
	}

	return 0;
}

static void __exit fan_rpm_exit(void)
{
	platform_driver_unregister(&fan_rpm_driver);
}

module_init(fan_rpm_init);
module_exit(fan_rpm_exit);

MODULE_AUTHOR("Linas Perkauskas <linas.perkauskas@teltonika.lt>");
MODULE_DESCRIPTION("Module for counting fan RPM from Frequecy Generator (FG) function");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:" DRV_NAME);
